// ==========================================================================
//
// = LIBRARY
//     PCIEFPGA
// 
// = FILENAME
//     PCIEFPGACSDat.cpp
// 
// = COPYRIGHT
//     (C) Copyright 2005 Agilent Technologies
//
// ==========================================================================
#include "PCIEFPGACSDat.h"
#include "devpcieintdefs.h"
#include "AgtPCIEControl.h"

#ifdef LINUXOFFLINE_DEBUG 
#define TRACE(msg) \
  {\
    fprintf(stderr,"%s\n",msg); \
  }
#else
#ifdef WIN32OFFLINE_DEBUG
#define TRACE(msg) \
  {\
    fprintf(stderr,"%s\n",msg); \
  }
#else
#define TRACE(msg)
#endif
#endif

#define DATA_MEM_SIZE_DW ((UInt32) (PCIE_DATAMEM_SIZE_GEN2_REL1 / 4))
#define COMPARE_DATA_EXPECTED_SIZE_DW ((UInt32) (PCIE_DATAMEM_COMPARE_DATA_EXPECTED_SIZE / 4))

#define MEM_COMP_BE_ADDR_BE_START_BIT    ((UInt8) 28)
#define MEM_COMP_BE_ADDR_BE_BIT_SIZE     ((UInt8)  4)
#define MEM_COMP_MISMATCH_ADDR_START_BIT ((UInt8)  0)
#define MEM_COMP_MISMATCH_ADDR_BIT_SIZE  ((UInt8) 21)
#define MEM_COMP_CTRL_RESET_BIT          ((UInt8)  1)
#define MEM_COMP_CTRL_REQUEST_SNAPSHOT_BIT ((UInt8) 0)

#define HOT_ACCESS_MEM_READ_WRITE_SELECT_BIT    ((UInt8) 31)
#define HOT_ACCESS_WRITE_BIT_VALUE              ((UInt8) 1)
#define HOT_ACCESS_READ_BIT_VALUE               ((UInt8) 0)

#define HOT_ACCESS_OFFSET_ADDR_SIZE             ((UInt8) 16)
#define HOT_ACCESS_OFFSET_ADDR_OFFSET_BIT       ((UInt8) 0)

#define PCIE_TIMOUT_FOR_DATAMEMWRITE  ((UInt32) 0x100000)
#define HOT_ACCESS_STATUS_DONE_BIT    ((UInt8) 0)

//***************************************************************************
CPCIEFpgaCSDat::CPCIEFpgaCSDat()
{
  TRACE( "CPCIEFpgaCSDat::CPCIEFpgaCSDat" );

  mHotAccessControl = PCIE_FPGA_CSDAT_OFFSET + PCIE_FPGAOFFSET_CSDAT_HOTACCESSCONTROL;
  mHotAccessDataBE  = PCIE_FPGA_CSDAT_OFFSET + PCIE_FPGAOFFSET_CSDAT_HOTACCESSDATABE;
  mHotAccessStatus  = PCIE_FPGA_CSDAT_OFFSET + PCIE_FPGAOFFSET_CSDAT_HOTACCESSSTATUS;
  mHotAccessRdDDw0  = PCIE_FPGA_CSDAT_OFFSET + PCIE_FPGAOFFSET_CSDAT_HOTACCESSRDDW0;
  mHotAccessWrDDw0  = PCIE_FPGA_CSDAT_OFFSET + PCIE_FPGAOFFSET_CSDAT_HOTACCESSWRDW0;
  /*mMemCompCtrl      = PCIE_FPGA_CSDAT_OFFSET + PCIE_FPGAOFFSET_CSDAT_MEMCOMPCTRL;  
  mMemCompStatus    = PCIE_FPGA_CSDAT_OFFSET + PCIE_FPGAOFFSET_CSDAT_MEMCOMPSTATUS;  
  mMemCompErrCount  = PCIE_FPGA_CSDAT_OFFSET + PCIE_FPGAOFFSET_CSDAT_MEMCOMPERRCOUNT;  
  mMemCompRefCount  = PCIE_FPGA_CSDAT_OFFSET + PCIE_FPGAOFFSET_CSDAT_MEMCOMPREFCOUNT;
  mMemCompActual    = PCIE_FPGA_CSDAT_OFFSET + PCIE_FPGAOFFSET_CSDAT_MEMCOMPACTUAL;  
  mMemCompExpected  = PCIE_FPGA_CSDAT_OFFSET + PCIE_FPGAOFFSET_CSDAT_MEMCOMPEXPECTED;  
  mMemCompBEAddr    = PCIE_FPGA_CSDAT_OFFSET + PCIE_FPGAOFFSET_CSDAT_MEMCOMPBEADDR; */

  mDataCmpIntAddr = 0;
  mDataCmpRealData = 0;
  mDataCmpExpectedData = 0;

  // make sure all the data is really written and read
  //  setHotAccessDataBE(portHandle, 0xffffffff);

  myController = new CAgtPCIEControl(); // returns only pointer if instance exists
}

//***************************************************************************
CPCIEFpgaCSDat::~CPCIEFpgaCSDat()
{
  delete myController;
  myController = NULL;
}

//
// register wrappers
//

// -----------------------------------------------------------------------
UInt32
CPCIEFpgaCSDat::getData(const AgtPortHandleT& portHandle, const UInt32& idx)
{
  if(idx < HOTACCESS_MEMSIZE_DW)
  {
    UInt32 offset = mHotAccessRdDDw0 + (4 * idx);
    UInt32 val;
    myController->FPGARead(portHandle,offset,val);
    return swapDW(val);
  }
  else
  {
    AGT_THROW("Invalid Read Data index ");
  }
}

// -----------------------------------------------------------------------
void
CPCIEFpgaCSDat::setData(const AgtPortHandleT& portHandle, const UInt32& idx, const UInt32& val)
{
  if(idx < HOTACCESS_MEMSIZE_DW)
  {
    UInt32 offset = mHotAccessWrDDw0 + (4 * idx);
    myController->FPGAWrite(portHandle,offset,swapDW(val));
  }
  else
  {
    AGT_THROW("Invalid Data index ");
  }
}

// -----------------------------------------------------------------------
//UInt32
//CPCIEFpgaCSDat::getMemCompCtrl(const AgtPortHandleT& portHandle)
//{
//  UInt32 val;
//  myController->FPGARead(portHandle,mMemCompCtrl,val);
//  return val;
//}
//
//// -----------------------------------------------------------------------
//void
//CPCIEFpgaCSDat::setMemCompCtrl(const AgtPortHandleT& portHandle, const UInt32& val)
//{
//  myController->FPGAWrite(portHandle,mMemCompCtrl,val);
//}
//
//// -----------------------------------------------------------------------
//UInt32
//CPCIEFpgaCSDat::getMemCompStatus(const AgtPortHandleT& portHandle)
//{
//  UInt32 val;
//  myController->FPGARead(portHandle,mMemCompStatus,val);
//  return val;
//}
//
//// -----------------------------------------------------------------------
//void
//CPCIEFpgaCSDat::setMemCompStatus(const AgtPortHandleT& portHandle, const UInt32& val)
//{
//  myController->FPGAWrite(portHandle,mMemCompStatus,val);
//}
//
//// -----------------------------------------------------------------------
//UInt32
//CPCIEFpgaCSDat::getMemCompErrCount(const AgtPortHandleT& portHandle)
//{
//  UInt32 val;
//  myController->FPGARead(portHandle,mMemCompErrCount,val);
//  return val;
//}
//
//// -----------------------------------------------------------------------
//UInt32
//CPCIEFpgaCSDat::getMemCompRefCount(const AgtPortHandleT& portHandle)
//{
//  UInt32 val;
//  myController->FPGARead(portHandle,mMemCompRefCount,val);
//  return val;
//}
//
//// -----------------------------------------------------------------------
//UInt32
//CPCIEFpgaCSDat::getMemCompActual(const AgtPortHandleT& portHandle)
//{
//  UInt32 val;
//  myController->FPGARead(portHandle,mMemCompActual,val);
//  return val;
//}
//
//// -----------------------------------------------------------------------
//UInt32
//CPCIEFpgaCSDat::getMemCompExpected(const AgtPortHandleT& portHandle)
//{
//  UInt32 val;
//  myController->FPGARead(portHandle,mMemCompExpected,val);
//  return val;
//}
//
//// -----------------------------------------------------------------------
//UInt32
//CPCIEFpgaCSDat::getMemCompBEAddr(const AgtPortHandleT& portHandle)
//{
//  UInt32 val;
//  myController->FPGARead(portHandle,mMemCompBEAddr,val);
//  return val;
//}

// -----------------------------------------------------------------------
UInt32
CPCIEFpgaCSDat::getHotAccessDataBE(const AgtPortHandleT& portHandle)
{
  UInt32 val;
  myController->FPGARead(portHandle,mHotAccessDataBE,val);
  return val;
}

// -----------------------------------------------------------------------
void
CPCIEFpgaCSDat::setHotAccessDataBE(const AgtPortHandleT& portHandle, const UInt32& val)
{
  myController->FPGAWrite(portHandle,mHotAccessDataBE,val);
}

// -----------------------------------------------------------------------
UInt32
CPCIEFpgaCSDat::getHotAccessControl(const AgtPortHandleT& portHandle)
{
  UInt32 val;
  myController->FPGARead(portHandle,mHotAccessControl,val);
  return val;
}

// -----------------------------------------------------------------------
void
CPCIEFpgaCSDat::setHotAccessControl(const AgtPortHandleT& portHandle, const UInt32& val)
{
  myController->FPGAWrite(portHandle,mHotAccessControl,val);
}

// -----------------------------------------------------------------------
UInt32
CPCIEFpgaCSDat::getHotAccessStatus(const AgtPortHandleT& portHandle)
{
  UInt32 val;
  myController->FPGARead(portHandle,mHotAccessStatus,val);
  return val;
}

//
// functions
//
    
// -----------------------------------------------------------------------
bool
CPCIEFpgaCSDat::compareData
(
 const AgtPortHandleT& portHandle,  
 const UInt32& offsetRealData, 
 const UInt32& sizeDW,
 const UInt8& firstDWBE,
 const UInt8& lastDWBE
)
{
  TRACE("CPCIEFpgaCSDat::compareData");
  
  if(offsetRealData % 4 == 0)
  {
    UInt32 dwOffsetExpectedData = (UInt32) ((offsetRealData - PCIE_DATAMEM_COMPARE_DATA_EXPECTED_SIZE) / 4);
    UInt32 firstMask = 0xffffffff;
    UInt32 lastMask = 0x0;
    
    if(firstDWBE != 0xf)
    {
      firstMask = 0x0;
      for(UInt8 delta = 0; delta< 4; delta++)
      {
        if( (firstDWBE >> delta) & 0x1)
        {
          firstMask |= 0xff << ((3 - delta) * 8);
        }
      }
    }
    
    if(checkBytesAtDWOffset(portHandle, dwOffsetExpectedData+COMPARE_DATA_EXPECTED_SIZE_DW,dwOffsetExpectedData,firstMask))
    {
      return 1; // error found
    }
    
    dwOffsetExpectedData++;
    
    if(sizeDW >= 2)
    {
      if(memCmp32(portHandle, dwOffsetExpectedData,dwOffsetExpectedData+COMPARE_DATA_EXPECTED_SIZE_DW,sizeDW-2))
      {
        for(UInt32 idx = 0; idx < (sizeDW - 2); idx ++) 
        {
          if(checkBytesAtDWOffset(portHandle, dwOffsetExpectedData+COMPARE_DATA_EXPECTED_SIZE_DW,dwOffsetExpectedData,0xffffffff))
          {
            return 1; // error found
          }
          dwOffsetExpectedData++;
        }
      }
      
      dwOffsetExpectedData += sizeDW-2;
      
      if(lastDWBE != 0x0)
      {
        for(UInt8 delta = 0; delta< 4; delta++)
        {
          if( (lastDWBE >> delta) & 0x1)
          {
            lastMask |= 0xff << ((3 - delta) * 8);
          }
        }
      }
      
      if(checkBytesAtDWOffset(portHandle, dwOffsetExpectedData+COMPARE_DATA_EXPECTED_SIZE_DW,dwOffsetExpectedData,lastMask))
      {
        return 1; // error found
      }
      
    }
    
  }

  return 0; // no error found
}

// -----------------------------------------------------------------------
UInt32
CPCIEFpgaCSDat::getCompareDataStatus(const DataCmpStatus& status)
{
  switch(status)
  {
  case RXDATAMEM_DATACMPSTATUS_INTADDR:
    return mDataCmpIntAddr;
  case RXDATAMEM_DATACMPSTATUS_REALDATA:
    return mDataCmpRealData;
  case RXDATAMEM_DATACMPSTATUS_EXPECTEDATA:
    return mDataCmpExpectedData;
  }

  return 0x0;
}


// -----------------------------------------------------------------------
void
CPCIEFpgaCSDat::writeData
(
  const AgtPortHandleT& portHandle, 
  const UInt32& address, 
  const UInt32& numBytes, 
  const UInt8* data, 
  UInt32& vectorOffsetIndex
)
{
  UInt32 currentAddress = address / 4;
  UInt32 byteCount = numBytes;
  UInt32 tmp = 0x0;
  
  // in case the address is not dw-aligned
  // this can only be true for the very first DW written
  if(address % 4)
  {
    // no read necessary as the mask is set now
    
    // this is the offset within the DWord where we start to write. It's 1,2 or 3
    UInt8 startOffset = (UInt8)(address % 4);

    // if the total number of bytes exceeds this DW, write the complete DW (maxData is 4)
    // if the total number of bytes fits into the DW, write only up to this position (startOffset+numBytes)
    UInt8 maxData = (UInt8)(byteCount >= (UInt8)(4 - startOffset) ? 4 : startOffset + byteCount);
    
    // fill the DW
    for(UInt8 i = startOffset; i < maxData; i++)
    {
      tmp &= ~(0xff <<   (i*8));
      tmp |= data[vectorOffsetIndex++] << (i*8);
      byteCount--;
    }

    // write the DW to HW
    setData(portHandle, currentAddress++,tmp);
  }
  
  // 0xab 0xff 0xcc 0x00 0x3e 0x33 0x12
  //             ^ 
  //   0   1     2    3    4    5    6
  // vectorOffsetIndex 

  while(byteCount > 4 )
  {
    tmp = 0;
    tmp |= data[vectorOffsetIndex++] <<  0;
    tmp |= data[vectorOffsetIndex++] <<  8;
    tmp |= data[vectorOffsetIndex++] <<  16;
    tmp |= data[vectorOffsetIndex++] <<  24;
    setData(portHandle, currentAddress++,tmp);
    byteCount -= 4;
  }
  
  // this is in case the very last DW is not completely filled
  tmp = 0;
  if(byteCount > 0)
  {
    // no read necessary as the mask is set now    
    
    // the numbers of bytes remaining
    UInt8 maxData = (UInt8)byteCount;

    // put the data into the DW
    for(UInt8 i = 0; i < maxData; i++)
    {
      tmp &= ~(0xff <<   (i*8));
      tmp |= data[vectorOffsetIndex++] << (i*8);
    }
    
    // write value into hw
    setData(portHandle, currentAddress,tmp);
  }
}

// -----------------------------------------------------------------------
UInt32
CPCIEFpgaCSDat::calculateMaskValueFromByteNum(UInt32 byteNum, UInt32 offset)
{
  if(byteNum >= 32)
  {
    return 0xffffffff;
  }

  if(byteNum == 0x0)
  {
    return 0x0;
  }
  
  UInt32 mask = 0x0;
  
  // for each byte set one bit
  for(UInt32 i=0 ; i < byteNum; i++)
  {
     mask |= 0x1 << i;  
  }

  // now shift the mask by offset
  mask <<= offset;

  // due to the byte swap, swap each nibble
  UInt32 newMask = 0x0;
  for(UInt32 posInDW = 0 ; posInDW < 32; posInDW += 4)
  {
    newMask |= ((mask >> (posInDW + 0)) & 0x1) << (posInDW + 3);
    newMask |= ((mask >> (posInDW + 1)) & 0x1) << (posInDW + 2);
    newMask |= ((mask >> (posInDW + 2)) & 0x1) << (posInDW + 1);
    newMask |= ((mask >> (posInDW + 3)) & 0x1) << (posInDW + 0);
  }

  return newMask;
}

// -----------------------------------------------------------------------
void
CPCIEFpgaCSDat::writeReadDataHotAccess(const AgtPortHandleT& portHandle, const UInt32& address, const UInt32& numBytes, UInt8* data, bool isWrite, AgtSizeT* pCount)
{
  UInt32 timeout = 0;
  
  UInt32 vectorOffsetIndex = 0;
  
  UInt32 numBytesChunk = 0;
  
  UInt32 controlRegisterVal = 0;
  
  if(isWrite)
  {
    setBits(controlRegisterVal,HOT_ACCESS_MEM_READ_WRITE_SELECT_BIT,1,HOT_ACCESS_WRITE_BIT_VALUE);
  }
  else
  {
    // always read each byte
    setHotAccessDataBE(portHandle, 0xffffffff);
    setBits(controlRegisterVal,HOT_ACCESS_MEM_READ_WRITE_SELECT_BIT,1,HOT_ACCESS_READ_BIT_VALUE);
  }
  
  if(numBytes == 0)
  {
    return;
  }
  
  UInt32 dwNum = 1 + ((numBytes - 1) / 4);
  UInt32 chunkNum = 1 + ((dwNum - 1) / HOTACCESS_MEMSIZE_DW);
  UInt32 addressChunk = address;
  UInt32 bytesTransferred = 0;
 
  for(UInt32 chunkIndex = 0; chunkIndex < chunkNum; chunkIndex++)
  {
    
    // this is the rest of bytes to be written
    // this is true for the chunk that is first and last
    // at the same time, and for the very last as well
    numBytesChunk =  numBytes - vectorOffsetIndex;
    
    // the rest can be 8DW at the max
    // this is true for all chunks that are neither 
    // the first nor the last
    if(numBytesChunk > (HOTACCESS_MEMSIZE_DW * 4))
    {
      numBytesChunk = (HOTACCESS_MEMSIZE_DW * 4);
    }
    
    // special case: For the first and not last chunk
    if(chunkIndex ==0 && chunkNum != 1)
    {
      numBytesChunk = 4 * (HOTACCESS_MEMSIZE_DW - 1) + addressChunk % 4;
      numBytesChunk += (addressChunk % 4 ? 0 : 4);
    }
    
    // select dw address
    setBits(controlRegisterVal,HOT_ACCESS_OFFSET_ADDR_OFFSET_BIT,HOT_ACCESS_OFFSET_ADDR_SIZE,addressChunk / 4);
     
    // the address must be 8DW aligned
    if(isWrite)
    {
      UInt32 mask = calculateMaskValueFromByteNum(numBytesChunk, addressChunk % 4 );
      setHotAccessDataBE(portHandle, mask);
      writeData(portHandle, addressChunk % 4,numBytesChunk,data,vectorOffsetIndex);
      setHotAccessControl(portHandle, controlRegisterVal);
    }
    else
    {
      setHotAccessControl(portHandle, controlRegisterVal);
      readData(portHandle, addressChunk % 4,numBytesChunk,data,vectorOffsetIndex);
    }
    
    // check for success
    while( timeout < PCIE_TIMOUT_FOR_DATAMEMWRITE && 
      (getHotAccessStatus(portHandle) & (0x1 << HOT_ACCESS_STATUS_DONE_BIT)) !=1 )
    {
      timeout++;
    }
    
    if(timeout >= PCIE_TIMOUT_FOR_DATAMEMWRITE)
    {
      // This is a timeout condition    
      return;
    }

    // update successfully transmitted data
    bytesTransferred += numBytesChunk;
    if(pCount)
    {
      *pCount = bytesTransferred;
    }

    //
    // prepare the next step
    //
    addressChunk += numBytesChunk;
  }
}


// -----------------------------------------------------------------------
void
CPCIEFpgaCSDat::writeDataHotAccess(const AgtPortHandleT& portHandle, const UInt32& address, const UInt32& numBytes, const UInt8* data)
{
  writeReadDataHotAccess(portHandle, address,numBytes,(UInt8*)data,1,NULL);   
}

// -----------------------------------------------------------------------
void
CPCIEFpgaCSDat::readDataHotAccess(const AgtPortHandleT& portHandle, const UInt32& address, const UInt32& numBytes, UInt8* data,  AgtSizeT* pCount)
{
  writeReadDataHotAccess(portHandle, address,numBytes,data,0,pCount); 
}

// -----------------------------------------------------------------------
void
CPCIEFpgaCSDat::readData(const AgtPortHandleT& portHandle, const UInt32& address, const UInt32& numBytes, UInt8* data, UInt32& vectorOffsetIndex)
{
  UInt32 currentAddress = address / 4;
  UInt32 byteCount = numBytes;

  // in case the address is not dw-aligned
  // this can only be true for the very first DW written  
  if(address % 4)
  {
    // get the 32bit value    
    UInt32 tmp = getData(portHandle, currentAddress);
    
    // this is the offset within the DWord where we start to write. It's 1,2 or 3    
    UInt8 startOffset = (UInt8)(address % 4);

    // if the total number of bytes exceeds this DW, write the complete DW (maxData is 4)
    // if the total number of bytes fits into the DW, write only up to this position (startOffset+numBytes)
    UInt8 maxData = (UInt8)(byteCount >= (UInt8)(4 - startOffset) ? 4 : startOffset + byteCount);
    
    // fill data vector with the data    
    for(UInt8 i = startOffset; i < maxData; i++)
    {
      data[vectorOffsetIndex++] = (UInt8)(tmp >> (i*8)) & 0xff;
      byteCount--;
    }
    currentAddress++;
  }
  
  // 0xab 0xff 0xcc 0x00 0x3e 0x33 0x12
  //             ^ 
  //   0   1     2    3    4    5    6
  // vectorOffsetIndex 

  while(byteCount > 4)
  {
    UInt32 tmp = getData(portHandle, currentAddress++);
    data[vectorOffsetIndex++] = (UInt8) ((tmp >>  0) & 0xff);
    data[vectorOffsetIndex++] = (UInt8) ((tmp >>  8) & 0xff);
    data[vectorOffsetIndex++] = (UInt8) ((tmp >> 16) & 0xff);
    data[vectorOffsetIndex++] = (UInt8) ((tmp >> 24) & 0xff);
    byteCount -= 4;
    
  }
  
  // this is in case we not reading all bytes of the last DW  
  if(byteCount > 0)
  {
    // read out the value of the last DW    
    UInt32 tmp = getData(portHandle, currentAddress);

    // the numbers of bytes remaining
    UInt8 maxData = (UInt8)byteCount;

    // get the data from the DW
    for(UInt8 i = 0; i < maxData; i++)
    {
      data[vectorOffsetIndex++] = (UInt8) ((tmp >> (i*8)) & 0xff);
    }
  }
}

// -----------------------------------------------------------------------
//bool
//CPCIEFpgaCSDat::getMemCompCtrlBit(const AgtPortHandleT& portHandle, const RxDataMemCompCtrlBit& theBit)
//{
//  UInt32 tmp = getMemCompCtrl(portHandle);
//  return (getBits(tmp,(UInt8)theBit,1) == 0 ? false : true);
//}
//
//// -----------------------------------------------------------------------
//void
//CPCIEFpgaCSDat::setMemCompCtrlBit(const AgtPortHandleT& portHandle, const RxDataMemCompCtrlBit& theBit, const bool& bVal)
//{
//  UInt32 tmp = getMemCompCtrl(portHandle);
//  setBits(tmp,(UInt8)theBit,1,bVal);
//  setMemCompCtrl(portHandle, tmp);
//}
//
//// -----------------------------------------------------------------------
//bool
//CPCIEFpgaCSDat::getMemCompStatusBit(const AgtPortHandleT& portHandle, const RxDataMemCompStatusBit& theBit)
//{
//  UInt32 tmp = getMemCompStatus(portHandle);
//  return (getBits(tmp,(UInt8)theBit,1) == 0 ? false : true);
//}
//
//// -----------------------------------------------------------------------
//void
//CPCIEFpgaCSDat::setMemCompStatusBit(const AgtPortHandleT& portHandle, const RxDataMemCompStatusBit& theBit, const bool& bVal)
//{
//  UInt32 tmp = getMemCompStatus(portHandle);
//  setBits(tmp,(UInt8)theBit,1,bVal);
//  setMemCompStatus(portHandle, tmp);
//}
//
//// -----------------------------------------------------------------------
//UInt32
//CPCIEFpgaCSDat::getMemCompMask(const AgtPortHandleT& portHandle)
//{
//  UInt32 mask = 0x0;
//  
//  UInt8 theBE = (UInt8) getBits(getMemCompBEAddr(portHandle),MEM_COMP_BE_ADDR_BE_START_BIT,MEM_COMP_BE_ADDR_BE_BIT_SIZE);
//   
//  for(UInt8 i=0;i< 4;i++)
//  {
//    if( (theBE >> i) & 0x1)
//    {
//      mask |= 0xff << (i * 8);
//    }
//  }
//  
//  return mask;
//}
//
//// -----------------------------------------------------------------------
//UInt8
//CPCIEFpgaCSDat::getMemCompActualByte(const AgtPortHandleT& portHandle)
//{
//
//  // swapDW fixes 
//  // SAD00253027 PCIE_EXERCISERSTATUS_DATAMEMCMP_INTADDR returns incorrect address for memory compare error  
//  UInt32 actual = swapDW(getMemCompActual(portHandle));
//  UInt32 expected = swapDW(getMemCompExpected(portHandle));
//  UInt8 theByte = 0x0;
//
//  for(UInt8 i=0;i < 4;i++)
//  {
//    theByte = (UInt8)((actual >> (i * 8)) & 0xff);
//    if( ((expected >> (i * 8)) & 0xff) != theByte)
//    {
//      return theByte;
//    }
//  } 
//
//  return 0x0;
//}
//
//// -----------------------------------------------------------------------
//UInt8
//CPCIEFpgaCSDat::getMemCompExpectedByte(const AgtPortHandleT& portHandle)
//{
//
//  // swapDW fixes 
//  // SAD00253027 PCIE_EXERCISERSTATUS_DATAMEMCMP_INTADDR returns incorrect address for memory compare error
//  UInt32 actual = swapDW(getMemCompActual(portHandle));
//  UInt32 expected = swapDW(getMemCompExpected(portHandle));
//  UInt8 theByte = 0x0;
//
//  for(UInt8 i=0;i < 4;i++)
//  {
//    theByte = (UInt8)((expected >> (i * 8)) & 0xff);
//    if( ((actual >> (i * 8)) & 0xff) != theByte)
//    {
//      return theByte;
//    }
//  } 
//
//  return 0x0;
//}
//
//// -----------------------------------------------------------------------
//UInt32
//CPCIEFpgaCSDat::getMemCompErrorAddress(const AgtPortHandleT& portHandle)
//{
//  UInt32 address = getBits(getMemCompBEAddr(portHandle),MEM_COMP_MISMATCH_ADDR_START_BIT,MEM_COMP_MISMATCH_ADDR_BIT_SIZE);
//  
//  // swapDW fixes 
//  // SAD00253027 PCIE_EXERCISERSTATUS_DATAMEMCMP_INTADDR returns incorrect address for memory compare error
//  UInt32 actual = swapDW(getMemCompActual(portHandle)); 
//  UInt32 expected = swapDW(getMemCompExpected(portHandle));
//  UInt8 theByte = 0x0;
//
//  for(UInt8 i=0;i < 4;i++)
//  {
//    theByte = (UInt8)((expected >> (i * 8)) & 0xff);
//    if( ((actual >> (i * 8)) & 0xff) != theByte)
//    {
//      return address + i;
//    }
//  } 
//
//  return 0x0;
//}

// -----------------------------------------------------------------------
// HELPER 
// -----------------------------------------------------------------------
// -----------------------------------------------------------------------
// -----------------------------------------------------------------------
bool
CPCIEFpgaCSDat::memCmp32(const AgtPortHandleT& portHandle, const UInt32& dwOffset1, const UInt32& dwOffset2, const UInt32& dwLen)
{
  for(UInt32 idx = 0; idx < dwLen; idx++)
  {
    UInt32 cmp1 = swapDW(getData(portHandle,(dwOffset1+idx)*4));
    UInt32 cmp2 = swapDW(getData(portHandle,(dwOffset2+idx)*4));
    if(cmp1 != cmp2)
    {
      return 1;
    }
  }

  return 0;
}

bool
CPCIEFpgaCSDat::checkBytesAtDWOffset
(
 const AgtPortHandleT& portHandle,  
 const UInt32& offsetRealData, 
 const UInt32& offsetExpectedData,
 const UInt32& mask
 )
{
//  TRACE("CPCIEFpgaCSDat::checkBytesAtDWOffset, RealData: " << offsetRealData << "Expected Data: " << offsetExpectedData);

  UInt32 cmp1 = mask & swapDW(getData(portHandle,offsetExpectedData*4));
  UInt32 cmp2 = mask & swapDW(getData(portHandle,offsetRealData*4));
  

  if(cmp1 != cmp2 )
  {
    mDataCmpIntAddr = offsetRealData * 4;
    mDataCmpRealData = swapDW(getData(portHandle,offsetRealData*4));
    mDataCmpExpectedData = swapDW(getData(portHandle,offsetExpectedData*4));

//    TRACE("CPCIEFpgaCSDat::checkBytesAtDWOffset Error Line A: IntAddr: " << mDataCmpIntAddr << ", RealData: " << mDataCmpRealData << ", Expected Data: " << mDataCmpExpectedData);
    

    // find culprit
    UInt32 deltaCount = 0;
    while(deltaCount < 4)
    {
      if( (mDataCmpRealData & (0xff << ((3-deltaCount) * 8))) != (mDataCmpExpectedData & (0xff << ((3-deltaCount) * 8))))
      {
        break;
      }
      deltaCount++;
    }
 
    mDataCmpIntAddr += deltaCount;
    mDataCmpRealData = (mDataCmpRealData >> ((3-deltaCount) * 8)) & 0xff;
    mDataCmpExpectedData = (mDataCmpExpectedData >> ((3-deltaCount) * 8)) & 0xff;

//    TRACE("CPCIEFpgaCSDat::checkBytesAtDWOffset Error Line B IntAddr: " << mDataCmpIntAddr << ", RealData: " << mDataCmpRealData << ", Expected Data: " << mDataCmpExpectedData);

    return 1;
  }
  else
  {
    return 0;
  }
}

// -----------------------------------------------------------------------
void
CPCIEFpgaCSDat::setBits(UInt32& val,const UInt8& bitPos,const UInt8& bitSize, const UInt32& bitVal)
{
  UInt32 mask;
  
  if(bitSize == 32)
  {
    val = bitVal;
  }
  else
  {
    mask = ((0x1 << bitSize) - 1);
    val &= ~(mask << bitPos);
    val |=  ((mask & bitVal) << bitPos);
  }
}
// -----------------------------------------------------------------------
UInt32
CPCIEFpgaCSDat::getBits(const UInt32& val,const UInt8& bitPos,const UInt8& bitSize)
{
  UInt32 mask;
  
  if(bitSize == 32)
  {
    return val;
  }
  else
  {
    mask = ((0x1 << bitSize) - 1);
    return (val >> bitPos) & mask;
  }
}


//---------------------------------------------------------------------------
UInt32
CPCIEFpgaCSDat::swapDW(const UInt32& val)
{
  return ((val & 0xff) << 24) | ((val & 0xff00) << 8) | ((val & 0xff0000) >> 8) | ((val & 0xff000000) >> 24);
}
